﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.IO;
using System.Reflection;
using RevisionAnalyser.Global;

namespace RevisionAnalyser
{
    public partial class MainForm
    {
        private SplashForm _splash;

        private void AppInit()
        {
            Update();
            DateTime startTime = DateTime.Now;
            Thread splashThread = new Thread(new ThreadStart(StartSplash));
            splashThread.Start();
            try
            {
                // Perform tasks
                InitUI();
                RegisterHotKeys();
                LoadPlugins();
                // Finish
                while (DateTime.Now.Subtract(startTime).TotalMilliseconds < 2000)
                {
                    Thread.Sleep(500);
                }
                CloseSplash();
            }
            catch (Exception ex)
            {
                TopMostMessageBox.Show("Critical error while initializing Revision Analyser, closing application.\n\nError: " + ex.Message, "Init error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                CloseSplash();
                Close();
            }
        }

        private void InitUI()
        {
            if (Directory.Exists(Constants.DEFAULT_PROJECTS_PATH))
            {
                uxOpenDialog.InitialDirectory = Constants.DEFAULT_PROJECTS_PATH;
            }
            else
            {
                uxOpenDialog.InitialDirectory = Constants.MY_DOCUMENTS_PATH;
            }
            Text = Constants.APP_TITLE;
        }

        private void RegisterHotKeys()
        {
            HotKeyRegister.RegisterHotKey(this, Keys.N | Keys.Control, HK_NEW);
            HotKeyRegister.RegisterHotKey(this, Keys.O | Keys.Control, HK_OPEN);
            HotKeyRegister.RegisterHotKey(this, Keys.Left | Keys.Control, HK_REV_PREV);
            HotKeyRegister.RegisterHotKey(this, Keys.Right | Keys.Control, HK_REV_NEXT);
            HotKeyRegister.RegisterHotKey(this, Keys.R | Keys.Control, HK_COLLAPSE_ALL);
            HotKeyRegister.RegisterHotKey(this, Keys.E | Keys.Control, HK_EXPAND_ALL);
            HotKeyRegister.RegisterHotKey(this, Keys.D1 | Keys.Control, HK_REV_LVL1);
            HotKeyRegister.RegisterHotKey(this, Keys.D2 | Keys.Control, HK_REV_LVL2);
            HotKeyRegister.RegisterHotKey(this, Keys.D3 | Keys.Control, HK_REV_LVL3);
            HotKeyRegister.RegisterHotKey(this, Keys.D4 | Keys.Control, HK_REV_LVL4);
            HotKeyRegister.RegisterHotKey(this, Keys.D5 | Keys.Control, HK_REV_LVL5);
            HotKeyRegister.RegisterHotKey(this, Keys.D6 | Keys.Control, HK_REV_LVL6);
            HotKeyRegister.RegisterHotKey(this, Keys.D7 | Keys.Control, HK_REV_LVL7);
            HotKeyRegister.RegisterHotKey(this, Keys.D8 | Keys.Control, HK_REV_LVL8);
            HotKeyRegister.RegisterHotKey(this, Keys.D9 | Keys.Control, HK_REV_LVL9);
        }

        private void LoadPlugins()
        {
            string[] plugins = Directory.GetFiles(Constants.PLUGINS_PATH, "*Project.dll");
            foreach (string plugin in plugins)
            {
                try
                {
                    String pluginName = Path.GetFileNameWithoutExtension(plugin);
                    String strippedName = pluginName.Replace("Project", string.Empty);
                    Assembly assembly = Assembly.LoadFile(plugin);
                    Type type = assembly.GetType(String.Format("RevisionAnalyser.Projects.{0}.{0}Type", pluginName));
                    ProjectTypes.Instance.AddProjectType(strippedName, type);
                }
                catch
                {
                    TopMostMessageBox.Show(String.Format("The plug-in '{0}' could not be loaded.\n\nPlease make sure this is a valid Revision Analyser plug-in.", plugin), "Plug-in error", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
            }
        }

        private void StartSplash()
        {
            _splash = new SplashForm();
            Application.Run(_splash);
        }

        private void CloseSplash()
        {
            if (_splash == null)
            {
                return;
            }
            _splash.Invoke(new EventHandler(_splash.KillMe));
            _splash.Dispose();
            _splash = null;
        }

        private void SetSplashStatus(string status)
        {
            if (_splash == null)
            {
                return;
            }
            _splash.SetStatus(status);
        }
    }
}
